
# 
# This file contains the the scripts which create the Apogee camera GUI. The procedures
# use the C++ API via a wrapper generated by SWIG.
#
#





#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : snapshot
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure is a minimal interface to take an exposure
#
#  Arguments  :
#
#               name	-	Image file name
#               exp	-	Exposure time in seconds
#               bcorr	-	Bias correction (1=yes) (optional, default is 0)
#               shutter	-	Shutter open(1), closed(0) (optional, default is 1)
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc snapshot { name exp {bcorr 0} {shutter 1} {id 0} } {
 
#
#  Globals    :
#  
#               STATUS	-	Exposure status
#               CFG	-	 
#               CAMERAS	-	Camera id's
#               DEBUG	-	Set to 1 for verbose logging
#               SCOPE	-	Telescope parameters, gui setup
global STATUS CFG CAMERAS DEBUG SCOPE ALTA
   if { $STATUS(busy) == 0 } {
    set STATUS(busy) 1
    set camera $CAMERAS($id)
    setutc
    if { $ALTA } {$camera read_ImagingStatus}
    $camera Expose $exp $shutter   
    if { $ALTA } {$camera read_ImagingStatus}
    set SCOPE(exposure) $exp
    set SCOPE(shutter) $shutter
    set SCOPE(exptype) Object
    if { $DEBUG } {debuglog "exposing (snapshot)"}
    after [expr $exp*1000+1000] "grabimage $name.fits $bcorr $id"
  }
}




#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : testgeometry
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure uses the Expose method to test that the current set
#  of geometry parameters represents a legal combination.
#
#  Arguments  :
#
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc testgeometry { {id 0} } {
 
#
#  Globals    :
#  
#               CONFIG	-	GUI configuration
#               CAMERAS	-	Camera id's
global CONFIG CAMERAS ALTA
   set exp 0.01
   set shutter 0
   set res "illegal"
   set camera $CAMERAS($id)
   if { $ALTA } {
     $camera UpdateGeneralStatus
     set res "ok"
   } else {
     set res [$camera Expose $exp $shutter]
   }
   return $res
}







#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : snapsleep
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure takes an exposure and then sleeps until we expect the
#  exposure is over. At which point it is read out. This routine is 
#  deprecated, as waitforimage is much more useful for normal usage.
#
#  Arguments  :
#
#               name	-	Image file name
#               exp	-	Exposure time in seconds
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
#               shutter	-	Shutter open(1), closed(0) (optional, default is 1)
 
proc snapsleep { name exp {id 0} {shutter 1} } {
 
#
#  Globals    :
#  
#               STATUS	-	Exposure status
#               CAMERAS	-	Camera id's
#               SCOPE	-	Telescope parameters, gui setup
global STATUS CAMERAS SCOPE ALTA DEBUG
    set camera $CAMERAS($id)
    setutc
    if { $ALTA } {$camera read_ImagingStatus}
    $camera Expose $exp $shutter   
    if { $ALTA } {$camera read_ImagingStatus}
    set SCOPE(exposure) $exp
    set SCOPE(shutter) $shutter
    if { $DEBUG } {debuglog "exposing (snapshot)"}
    set STATUS(busy) 1
    exec sleep [expr ($exp*10+1000)/1000 + 1] 
    grabimage $name.fits
}




proc abortexposure { {id 0} } {
global CAMERAS ALTA STATUS
  set camera $CAMERAS($id)
  if { $ALTA } {
    $camera StopExposure 0
    $camera write_ForceShutterOpen 0
  } else {
    $camera write_ForceShutterOpen 0
    $camera write_Shutter 0
    $camera Flush
  }
  set STATUS(busy) 0
}


proc getaltastatus { {id 0} } {
global CAMERAS CREAD CVARS
  set c $CAMERAS($id)
  set i read_ImagingStatus
#  puts stdout "[string range $i 5 end] =          [$c $i]"
  return [$c $i]
}


proc waitforalta { {n 100} {id 0} } {
global CAMERA tcl_platform
  set i 0
  while { $i < $n } {
    set r [$CAMERA read_ImagingStatus]
    set ir [$CAMERA ImageReady]
    if { $ir == 1 } {
 puts stdout "waited for $i"
       return 5
    }
    if { $tcl_platform(os) == "Darwin" } {
      exec sleep 1
    } else {
      exec usleep 10000
    }
    update idletasks
    incr i 1
  }
  return 0
}


#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : waitforimage
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure periodically wakes up and checks if the current exposure is
#  ready to be read out yet. If not, it loops around again. If the elapsed time
#  exceeds the expected exposure time (plus a couple of seconds) then it times out.
#
#  Arguments  :
#
#               exp	-	Exposure time in seconds
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc waitforimage { exp {id 0} } {
 
#
#  Globals    :
#  
#               CAMERAS	-	Camera id's
#               STATUS	-	Exposure status
#               DEBUG	-	Set to 1 for verbose logging
global CAMERAS STATUS DEBUG SCOPE ALTA REMAINING tcl_platform
  set camera $CAMERAS($id)
  set REMAINING $exp
  exec sleep 1
  if { $exp == 0 } {set exp 1}
  set STATUS(readout) 0
  set STATUS(pause) 0
  set SCOPE(darktime) 0
  update idletasks
  if { $ALTA } {
    set lookfor 5
    set s 0
  } else {
    set s [$camera read_Status]
    set lookfor 5
  }
  while { $s != $lookfor && $exp > 0} {
     update 
     if { $tcl_platform(os) == "Darwin" } {
       exec sleep 1
     } else {
       exec usleep 990000
     }
     update idletasks
     incr exp -1
     set REMAINING $exp
     if { $ALTA } {
       set s 0
     } else {
       set s [$camera read_Status]
     }
     if { $STATUS(abort) } {
        abortexposure
        set exp -1
        return 1
     }
     if { $STATUS(pause) } {
        $camera write_ForceShutterOpen 0
        if { $ALTA == 0 } {$camera write_Shutter 0}
        while { $STATUS(pause) } {
           if { $STATUS(abort) } {
             abortexposure
             set exp -1
             return 1
           }
           update 
           if { $tcl_platform(os) == "Darwin" } {
             exec sleep 1
           } else {
             exec usleep 100000
           }
           set SCOPE(darktime) [expr $SCOPE(darktime) +1]
        }
        $camera write_ForceShutterOpen 1
        if { $ALTA == 0 } {$camera write_Shutter 1}
     }
     if { $SCOPE(darktime) > 0 } {set s 0}
     if { $DEBUG } {debuglog "waiting $exp $s"}
  }
  if { $ALTA } {set s [waitforalta]}
  if { $SCOPE(darktime) > 0 } {
     if { $ALTA == 0 } {$camera write_Shutter 0}
     $camera write_ForceShutterOpen 0
     set s $lookfor
  }
  set STATUS(readout) 1
  update
  if { $s != $lookfor } {
     return 1
  }
  return 0
}






#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : obstodisk
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure performs the standard exposure operations. The image is
#  saved to a FITS file on disk.
#
#  Arguments  :
#
#               n	-	Number of frame(s)
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc obstodisk { n {id 0} } {
 
#
#  Globals    :
#  
#               CAMERAS	-	Camera id's
#               STATUS	-	Exposure status
#               SCOPE	-	Telescope parameters, gui setup
#               DEBUG	-	Set to 1 for verbose logging
global CAMERAS STATUS SCOPE DEBUG ALTA REMAINING
  if { $SCOPE(comptimer) } {
    timerobstodisk $n $id
  } else {
    set camera $CAMERAS($id)
    set STATUS(busy) 1
    setutc
    if { $ALTA } {$camera UpdateGeneralStatus}
    if { [expr int($SCOPE(exposure))] > 3 } { 
       set REMAINING [expr int($SCOPE(exposure))]
       countdown [expr int($SCOPE(exposure))]
    }
    $camera Expose $SCOPE(exposure) $SCOPE(shutter)
    if { $ALTA } {$camera UpdateGeneralStatus}
    if { $DEBUG } {debuglog "exposing (obstobuffer)"}
    set timeout [waitforimage [expr int($SCOPE(exposure))] $id]
    if { $timeout } {
       puts stdout "TIMEOUT/ABORT"
    } else {   
      if { $DEBUG } {debuglog "Reading out..."}
      set d1 [exec date]
      $camera BufferImage tempobs
      set d2 [exec date]
      puts stdout "$d1 $d2"
      countdown off
      set STATUS(readout) 0
      if { [llength [split $SCOPE(imagename) "\%"]] > 1 } {
         set name "$SCOPE(datadir)/[format "$SCOPE(imagename)" $SCOPE(seqnum)].fits"
         incr SCOPE(seqnum) 1
      } else {
         set name "$SCOPE(datadir)/$SCOPE(imagename).fits"
      }
      saveandshow tempobs $name
   }
 }
}





#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : saveandshow
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure is used to copy an image from an in-memory buffer to 
#  a FITS file on disk. It optionally deletes the file first if
#  overwrite is enabled. It also calls the appropriate routine for 
#  bias correction if that is enabled, and finally displays the image
#  in DS9 if that is enabled.
#
#  Arguments  :
#
#               buffer	-	Name of in-memory image buffer
#               name	-	Image file name
 
proc saveandshow { buffer name } {
 
#
#  Globals    :
#  
#               CAMERAS	-	Camera id's
#               STATUS	-	Exposure status
#               SCOPE	-	Telescope parameters, gui setup
#               DEBUG	-	Set to 1 for verbose logging
global CAMERAS STATUS SCOPE DEBUG
      if { [file exists $name] } {
         if { $SCOPE(overwrite) } {
            exec rm -f $name
         } else {
            set it [ tk_dialog .d "File exists" "The file named\n $name\n already exists, Overwrite it ?" {} -1 No "Yes"]           
            if { $it } {
               exec rm -f $name
            } else {
                set saveas [tk_getSaveFile -initialdir [file dirname $name] -filetypes {{{FITS images} {.fits}}}]
                set name [file rootname $saveas].fits
                saveandshow $buffer $name
                return
            }
         }
      }
      if { $SCOPE(autocalibrate) } {
          loadcalibrations
      }
      if { $SCOPE(autobias) } {
         write_cimage $buffer $SCOPE(exposure) $name
      } else {
         write_calibrated $buffer $SCOPE(exposure) $name 0
#           write_image $buffer $name
      }
      if { $SCOPE(autodisplay) } {
        checkDisplay
        exec xpaset -p ds9 file $name
      } 
      set STATUS(busy) 0
}





#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : grabimage
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This routine  readouts the image from the chip, and then writes it
#  to a disk FITS file. Either raw or bias corrected images are supported.
#  
#  Arguments  :
#
#               name	-	Image file name
#               bcorr	-	Bias correction (1=yes) (optional, default is 0)
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc grabimage { name {bcorr 0} {id 0} } {
 
#
#  Globals    :
#  
#               STATUS	-	Exposure status
#               CAMERAS	-	Camera id's
#               DEBUG	-	Set to 1 for verbose logging
#               SCOPE	-	Telescope parameters, gui setup
global STATUS CAMERAS DEBUG SCOPE
    set camera $CAMERAS($id)
    if { $DEBUG } {debuglog "Reading out..."}
    $camera BufferImage READOUT
    if { [file exists $name] } {
      puts stdout "Overwriting $name"
      exec rm -f $name
    }
    if { $DEBUG } {debuglog "Saving to FITS $name"}
    if { $bcorr } {
       write_cimage READOUT $SCOPE(exposure) $name
    } else {
       write_image READOUT  $name
    }
    if { $SCOPE(autodisplay) } {
      checkDisplay
      exec xpaset -p ds9 file $name
    }
    set STATUS(busy) 0
}





#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : displayimage
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This routine uses the XPA interface to request a shared memory transfer
#  of image data to the DS9 image display tool.
#
#  Arguments  :
#
#               name	-	Image file name
 
proc displayimage { name } {
 
#
#  Globals    :
#  
  set pars [shmem_image $name]
  set cmd "exec xpaset -p ds9  shm array shmid [lindex $pars 0] [lindex $pars 1] \\\[xdim=[lindex $pars 2],ydim=[lindex $pars 3],bitpix=16\\\]"
  eval $cmd
}




#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : abortsequence
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure aborts the current exposure or sequence of exposures.
#  It simply sets the global abort flag and resets the GUI widgets.
#
#  Arguments  :
#
 
proc abortsequence { } {
 
#
#  Globals    :
#  
#               STATUS	-	Exposure status
global STATUS
  set STATUS(abort) 1
  countdown off
  .main.observe configure -text "Observe" -bg gray -relief raised
  .main.abort configure -bg gray -relief sunken -fg LightGray
  .main.pause configure -bg gray -relief sunken -fg LightGray
  .main.resume configure -bg gray -relief sunken -fg LightGray
}





#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : continuousmode
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure continuously calls itself to repeatedly take exposures
#  and auto-display them. It will generally be used to image acquisition
#  and focus applications.
#  This mode of operation will continue until the user clicks "abort"
#
#  Arguments  :
#
#               exp	-	Exposure time in seconds
#               n	-	Number of frame(s) (optional, default is 9999)
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc continuousmode { exp {n 9999} {id 0} } {
 
#
#  Globals    :
#  
#               STATUS	-	Exposure status
#               CAMERAS	-	Camera id's
global STATUS CAMERAS SCOPE ALTA
   if { $STATUS(abort) } {set STATUS(abort) 0 ; return}
   if { $STATUS(busy) } {after 100 continuousmode $exp $n}
   .main.observe configure -text "continuous" -bg green -relief sunken
   .main.abort configure -bg orange -relief raised -fg black
   .main.pause configure -bg orange -relief raised -fg black
   update
   set camera $CAMERAS($id)
   exec rm -f /tmp/continuous.fits
   if { $ALTA } {$camera read_ImagingStatus}
   $camera Expose $exp 1
   if { $ALTA } {$camera read_ImagingStatus}
   waitforimage $exp
   $camera BufferImage READOUT
###   write_calibrated READOUT $SCOPE(exposure) /tmp/continuous.fits 0
###   checkDisplay
###   exec xpaset -p ds9 file  /tmp/continuous.fits
     displayimage READOUT
   incr n -1
   if { $n > 0 } {
      after 10 continuousmode $exp $n
   } else {
      .main.observe configure -text "Observe" -bg gray -relief raised
      .main.abort configure -bg gray -relief sunken -fg LightGray
      .main.pause configure -bg gray -relief sunken -fg LightGray
   }
}




#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : observe
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This stub routine responds to user selections on the observe menu.
#
#  Arguments  :
#
#               op	-	Operation specifier
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc observe { op {id 0} } {
 
#
#  Globals    :
#  
#               SCOPE	-	Telescope parameters, gui setup
global SCOPE
  switch $op {
      region {acquisitionmode}
      multiple {continuousmode $SCOPE(exposure) 9999 $id}
      fullframe {setfullframe}
  }
}




#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : setfullframe
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This stub routine responds to user selections on the observe menu.
#
#  Arguments  :
#
#               op	-	Operation specifier
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc setfullframe { } {
 
#
#  Globals    :
#  
#               SCOPE	-	Telescope parameters, gui setup
global SCOPE CONFIG
   set CONFIG(geometry.StartCol)  $SCOPE(StartCol)
   set CONFIG(geometry.StartRow)  $SCOPE(StartRow)
   set CONFIG(geometry.NumCols) $SCOPE(NumCols)
   set CONFIG(geometry.NumRows) $SCOPE(NumRows)
}






#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : acquisitionmode
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure controls the specification of a sub-image region using
#  the DS9 image display tool.
#
#  Arguments  :
#
 
proc  acquisitionmode { } {
 
#
#  Globals    :
#  
#               ACQREGION	-	Sub-frame region coordinates
#               CONFIG	-	GUI configuration
global ACQREGION CONFIG
  set it [ tk_dialog .d "Acquisition region" "Click New to define a new region,\n OK to use the current region " {} -1 OK "New"]      
  if {$it} {
    catch {
      exec xpaset -p ds9 regions deleteall
      exec xpaset -p ds9 regions box $ACQREGION(xs) $ACQREGION(ys) $ACQREGION(xe) $ACQREGION(ye)
    }
    set it [tk_dialog .d "Edit region" "Resize/Move the region in the\n image display tool then click OK" {} -1 "OK"]
    set reg [split [exec xpaget ds9 regions] \n]
    foreach i $reg {
     if { [string range $i 0 8] == "image;box" } {
        set r [lrange [split $i ",()"] 1 4]
        set ACQREGION(xs) [expr int([lindex $r 0] - [lindex $r 2]/2)]
        set ACQREGION(ys) [expr int([lindex $r 1] - [lindex $r 3]/2)]
        set ACQREGION(xe) [expr int([lindex $r 0] + [lindex $r 2]/2)]
        set ACQREGION(ye) [expr int([lindex $r 1] + [lindex $r 3]/2)]
        puts stdout "selected region $r"
     }
    }
  } 
  set CONFIG(geometry.StartCol) [expr $ACQREGION(xs)]
  set CONFIG(geometry.StartRow) [expr $ACQREGION(ys)]
  set CONFIG(geometry.NumCols) [expr $ACQREGION(xe)-$ACQREGION(xs)+1]
  set CONFIG(geometry.NumRows) [expr $ACQREGION(ye)-$ACQREGION(ys)+1]
}






#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : countdown
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This routine manages a countdown window. The window displays the 
#  current frame number, and seconds remaining.
#
#  Arguments  :
#
#               time	-	Countdown time in seconds
 
proc countdown { time } {
 
#
#  Globals    :
#  
#               FRAME	-	Frame number in a sequence
#               STATUS	-	Exposure status
global FRAME STATUS REMAINING
  if { $time == "off" || $STATUS(abort) } {
     wm withdraw .countdown
     return
  }
  .countdown.f configure -text $FRAME
  .countdown.t configure -text $REMAINING
  if { $STATUS(pause) == 0 } {
      set time [expr $time - 1]
  } else {
      .countdown.t configure -text "$time (HOLD)"
  }
  if { [winfo ismapped .countdown] == 0 } {
     wm deiconify .countdown
     wm geometry .countdown +20+20
  }
  if { $time > -1 } {
     update
     after 850 countdown $time
  } else {
     if { $STATUS(readout) } {
       .countdown.t configure -text "READING"
     } else {
       wm withdraw .countdown
     }
  }
}




#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : startsequence
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This routine manages a sequence of exposures. It updates bias columns
#  specifications in case they have been changed, then it loops thru
#  a set of frames, updating the countdown window, and calling obstodisk to 
#  do the actual exposures.
#
#  Arguments  :
#
 
proc startsequence { } {
 
#
#  Globals    :
#  
#               SCOPE	-	Telescope parameters, gui setup
#               OBSPARS	-	Default observation parameters
#               FRAME	-	Frame number in a sequence
#               STATUS	-	Exposure status
#               DEBUG	-	Set to 1 for verbose logging
global SCOPE OBSPARS FRAME STATUS DEBUG
   set OBSPARS($SCOPE(exptype)) "$SCOPE(exposure) $SCOPE(numframes) $SCOPE(shutter)"
   set STATUS(abort) 0
   if { $SCOPE(lobias) > 0 && $SCOPE(hibias) > 0 } {
      set_biascols $SCOPE(lobias) $SCOPE(hibias)
   }
   .main.observe configure -text "working" -bg green -relief sunken
   .main.abort configure -bg orange -relief raised -fg black
   .main.pause configure -bg orange -relief raised -fg black
   set i 1
   while { $i <= $SCOPE(numframes) && $STATUS(abort) == 0 } {
      set FRAME $i
      countdown [expr int($SCOPE(exposure))]
      if { $DEBUG} {debuglog "$SCOPE(exptype) frame $i"}
      obstodisk $i 
      incr i 1
   }
   .main.observe configure -text "Observe" -bg gray -relief raised
   .main.abort configure -bg gray -relief sunken -fg LightGray
   .main.pause configure -bg gray -relief sunken -fg LightGray
   countdown off
}


proc pausesequence { } {
global STATUS
  set STATUS(pause) 1
  .main.pause configure -bg gray -relief sunken -fg Black -bg yellow
  .main.resume configure -bg gray -relief raised -fg Black -bg orange
}

proc resumesequence { } {
global STATUS
  set STATUS(pause) 0
  .main.pause configure -bg orange -relief raised -fg black
  .main.resume configure -bg gray -relief sunken -fg LightGray
}



#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : driftscan
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure manages driftscan observations. All the heavy lifting is
#  done by the C/C++ code of course.
#
#  Arguments  :
#
#               nrow	-	Number of rows to readout (optional, default is 0)
#               delay	-	Per-row delay in microseconds (optional, default is 0)
#               nblock	-	Number of rows to read per cycle (optional, default is 1)
#               shutter	-	Shutter open(1), closed(0) (optional, default is 1)
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc driftscan { {nrow 0} {delay 0} {nblock 1} {shutter 1} {id 0} } {
 
#
#  Globals    :
#  
#               CAMERAS	-	Camera id's
#               SCOPE	-	Telescope parameters, gui setup
#               DEBUG	-	Set to 1 for verbose logging
global CAMERAS SCOPE DEBUG
  if { $delay == 0 } {set delay $SCOPE(driftdcalc)}
  if { $nrow == 0 } {set nrow $SCOPE(driftrows)}
  if { $DEBUG } {debuglog  "Drift scan using delay = $delay, rows = $nrow"}
  set camera $CAMERAS($id)
  setutc
  $camera configure -m_TDI 1
  $camera Expose [expr $delay/1000] $shutter
  $camera BufferDriftScan drift $delay $nrow $nblock 0
  set name "$SCOPE(datadir)/$SCOPE(imagename).fits"
  saveandshow drift $name
  $camera configure -m_TDI 0
  $camera Flush
}





#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : driftcalib
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#
#  This procedure calculates the equatorial drift rate based upon a 
#  rate measured by the user at an arbritary (hopefully accurate) known
#  DEC.
#
#  Arguments  :
#
 
proc driftcalib { } {
 
#
#  Globals    :
#  
#               SCOPE	-	Telescope parameters, gui setup
global SCOPE
   set delay $SCOPE(driftsamp)
   set d [dms_to_radians [.main.vdec get]]
   set fac [expr cos($d)]
   set SCOPE(driftdelay) [expr int($delay*$fac)]
}




#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : driftcalc
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure calculates the drift rate in microseconds. It presumes that
#  the equatorial rate has already been calculated.
#
#
#  Arguments  :
#
 
proc driftcalc { } {
 
#
#  Globals    :
#  
#               SCOPE	-	Telescope parameters, gui setup
#               CAMSTATUS	-	Current values of camera variables
global SCOPE CAMSTATUS
   set delay $SCOPE(driftdelay)
   set d [dms_to_radians [.main.vdec get]]
   set fac [expr cos($d)]
   set SCOPE(driftdcalc) [expr int($delay/$fac)]
   set t [tlabel [expr $SCOPE(driftdcalc)/1000000.*($CAMSTATUS(NumY)+$SCOPE(driftrows))/3600./12.*3.14159]]
   set SCOPE(driftexp) $t
}




#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : focustest
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure takes a focus exposure. This consists of taking a series
#  of exposures, and shifting the charge on the ccd in between each.
#  In addition, the focus is also altered between each exposure.
#  The resulting image can be automatically analsed to determine 
#  optimum focus.
#  This is an example of using TDI mode, without writing any c/c++ code, the 
#  applications of this technique are legion.....
#
#  Arguments  :
#
#               name	-	Image file name
#               exp	-	Exposure time in seconds
#               nstep	-	Number of steps in focus
#               nshift	-	Number of rows to shift per focus step
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc focustest { name exp nstep nshift {id 0} } {
 
#
#  Globals    :
#  
#               CAMERAS	-	Camera id's
#               DEBUG	-	Set to 1 for verbose logging
#               CAMSTATUS	-	Current values of camera variables
global CAMERAS DEBUG CAMSTATUS
  set camera $CAMERAS($id)
  setutc
  $camera configure -m_TDI 1
  $camera Expose 0.02 0
  set istep 1
  set ishift 0
  while { $ishift < $CAMSTATUS(NumY) } {
     $camera DigitizeLine
     incr ishift 1
  }
  while { $istep <= $nstep } {
    set it [tk_dialog .d "Focus exposure" "Move to focus position $istep\n then click OK" {} -1 "Cancel" "OK"]
    if { $it } {
      if { $istep == $nstep } {set nshift [expr $nshift*2] }
      $camera write_ForceShutterOpen 1 
      exec sleep $exp
      $camera write_ForceShutterOpen 0
      set ishift 0
      while { $ishift < $nshift } {
         $camera DigitizeLine
         incr ishift 1
      }
    } else { 
       set istep $nstep
    }
    incr istep 1
  }
  $camera configure -m_TDI 0
  $camera BufferImage READOUT
  $camera Flush
  if { $DEBUG } {debuglog "Saving to FITS $name"}
  write_image READOUT $name.fits
}





#---------------------------------------------------------------------------
#---------------------------------------------------------------------------
#
#  Procedure  : timedobstodisk
#
#---------------------------------------------------------------------------
#  Author     : Dave Mills (djm@randomfactory.com)
#  Version    : 0.9
#  Date       : Aug-04-2003
#  Copyright  : The Random Factory, Tucson AZ
#  License    : GNU GPL
#  Changes    :
#
#  This procedure performs the standard exposure operations. The image is
#  saved to a FITS file on disk.
#
#  Arguments  :
#
#               n	-	Number of frame(s)
#               id	-	Camera id (for multi-camera use) (optional, default is 0)
 
proc timerobstodisk { n {id 0} } {
 
#
#  Globals    :
#  
#               CAMERAS	-	Camera id's
#               STATUS	-	Exposure status
#               SCOPE	-	Telescope parameters, gui setup
#               DEBUG	-	Set to 1 for verbose logging
global CAMERAS STATUS SCOPE DEBUG ALTA
    set camera $CAMERAS($id)
    set STATUS(busy) 1
    setutc
#    $camera configure -m_TDI 1
    $camera Expose 0.02 0
    if { $SCOPE(shutter) } {
        $camera write_ForceShutterOpen 1 
    }
    if { $DEBUG } {debuglog "exposing (obstobuffer)"}
    if { [expr int($SCOPE(exposure))] > 3 } { 
       countdown [expr int($SCOPE(exposure))]
    }
    set timer [expr [clock seconds] + $SCOPE(exposure)]
    while { [expr  $timer-[clock seconds]] > 0 } {
        exec usleep 100 
        update
    }
     $camera write_ForceShutterOpen 0
#    $camera configure -m_TDI 0
#    if { $timeout } {
#       puts stdout "TIMEOUT/ABORT"
#    } else {   
      if { $DEBUG } {debuglog "Reading out..."}
      set d1 [exec date]
      $camera BufferImage tempobs
      if { $ALTA == 0 } {$camera Flush}
      set d2 [exec date]
      puts stdout "$d1 $d2"
      countdown off
      set STATUS(readout) 0
      if { [llength [split $SCOPE(imagename) "\%"]] > 1 } {
         set name "$SCOPE(datadir)/[format "$SCOPE(imagename)" $SCOPE(seqnum)].fits"
         incr SCOPE(seqnum) 1
      } else {
         set name "$SCOPE(datadir)/$SCOPE(imagename).fits"
      }
      saveandshow tempobs $name
#   }
}





