/*! 
* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this file,
* You can obtain one at http://mozilla.org/MPL/2.0/.
*
* Copyright(c) 2009 Apogee Instruments, Inc. 
*
* \class Alta 
* \brief Derived class for the alta apogee cameras 
* 
*/ 

#ifndef ALTA_INCLUDE_H__ 
#define ALTA_INCLUDE_H__ 

#include <string>

#include "ApogeeCam.h" 


class DLL_EXPORT Alta : public ApogeeCam
{ 
    public: 
        Alta();

        virtual ~Alta(); 

        void OpenConnection( const std::string & ioType,
             const std::string & DeviceAddr,
             const uint16_t FirmwareRev,
             const uint16_t Id );

        void Init();

        void StartExposure( double Duration, bool IsLight );

        CameraStatusRegs GetStatus();
        Apg::Status GetImagingStatus();
      
        void GetImage( std::vector<uint16_t> & out );

        void StopExposure( bool Digitize );

        uint32_t GetAvailableMemory();

        /*! 
         * Sets the analog to digital converter gain value for the 12 bit ADC.
         * \param [in] gain The new gain value. 0-1023 is a valid range.
         * Calling ApogeeCam::SetAdcGain( gain, 1, 0 ) is equivalent to this function
         * and is recommended for use.
         * \exception std::runtime_error
         */
        void SetCcdAdc12BitGain( uint16_t gain );

         /*! 
         * Sets the analog to digital converter offset vaule for the 12 bit ADC.
         * \param [in] gain The new offset value. 0-255 is a valid range. 
         * Calling ApogeeCam::SetAdcOffset( offset, 1, 0 ) is equivalent to this function
         * and is recommended for use.
         * \exception std::runtime_error
         */
        void SetCcdAdc12BitOffset( uint16_t offset );    

        /*! 
         * Returns the analog to digital converter gain value for the 12 bit ADC.
         * Calling ApogeeCam::GetAdcGain( 1, 0 ) is equivalent to this function
         * and is recommended for use.
         * \exception std::runtime_error
         */
        uint16_t GetCcdAdc12BitGain();

         /*! 
         * Returns the analog to digital converter offset value for the 12 bit ADC.
         * Calling ApogeeCam::GetAdcOffset( 1, 0 ) is equivalent to this function
         * and is recommended for use.
         * \exception std::runtime_error
         */
        uint16_t GetCcdAdc12BitOffset();

        /*! 
         * Returns the analog to digital converter gain value for the 16 bit ADC.
         * Calling ApogeeCam::GetAdcOffset( 0, 0 ) is equivalent to this function
         * and is recommended for use.
         * \exception std::runtime_error
         */
        double GetCcdAdc16BitGain();

        int32_t GetNumAds();
        int32_t GetNumAdChannels();

        double GetCoolerDrive();

        /*! 
         *
         * \exception std::runtime_error
         */
        void SetFanMode( Apg::FanMode mode, bool PreCondCheck = true );

        /*! 
         *
         * \exception std::runtime_error
         */
        Apg::FanMode GetFanMode();

        double GetTempHeatsink();

		/*! 
         * Returns an ethernet's camera MAC address.  Will throw
         * an std::runtime_error exception if the call is made on a
         * USB camera.
         * \exception std::runtime_error
         */
        std::string GetMacAddress();

    protected:
        Alta(const std::string & ioType,
             const std::string & DeviceAddr);

        void ExposureAndGetImgRC(uint16_t & r, uint16_t & c);
        uint16_t ExposureZ();
        uint16_t GetImageZ();
        uint16_t GetIlluminationMask();
        void CreateCamIo(const std::string & ioType,
            const std::string & DeviceAddr);

        void FixImgFromCamera( const std::vector<uint16_t> & data,
            std::vector<uint16_t> & out,  int32_t rows, int32_t cols);

    private:
        
        void VerifyCamId();
        void CfgCamFromId( uint16_t CameraId );
        uint16_t GetPixelShift();

        void Init12BitCcdAdc();
        void StopExposureModeTdiKinetics( bool Digitize );
        

        const std::string m_fileName;
        
        //disabling the copy ctor and assignment operator
        //generated by the compiler - don't want them
        //Effective C++ Item 6
        Alta(const Alta&);
        Alta& operator=(Alta&);
}; 

#endif
